<?php

namespace App\Http\Controllers;

use App\Models\Customer;
use App\Models\ProductMaster;
use App\Models\Quotation;
use App\Models\QuotationItem;
use App\Models\TermsCondition;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Http\Request;

class QuotationController extends Controller
{
    public function index()
    {
        $quotations = Quotation::with('customer')->latest()->paginate(10);
        return view('quotations.index', compact('quotations'));
    }

    public function create()
    {
        $customers = Customer::all();
        $termsConditions = TermsCondition::where('is_active', true)->orderBy('display_order')->get();
        // Get grouped products (models with their specs)
        $allProducts = \App\Models\Product::with('master')->get();
        $products = $allProducts->groupBy(function ($product) {
            return $product->product_master_id . '-' . $product->product_model . '-' . $product->price;
        })->map(function ($group) {
            $first = $group->first();
            $specs = $group->map(function ($product) {
                return [
                    'spec_name' => $product->spec_name,
                    'spec_value' => $product->spec_value,
                    'spec_unit' => $product->spec_unit,
                ];
            })->toArray();
            
            return [
                'id' => $first->id,
                'product_master_id' => $first->product_master_id,
                'product_model' => $first->product_model,
                'price' => $first->price,
                'master' => $first->master,
                'specs' => $specs,
                'spec_count' => count($specs),
                'product_name' => $first->master->product_name . ' (' . $first->product_model . ')',
            ];
        })->values();
        
        return view('quotations.create', compact('customers', 'products', 'termsConditions'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'customer_id' => 'required|exists:customers,id',
            'quotation_date' => 'required|date',
            'items' => 'required|array|min:1',
            'items.*.product_id' => 'required|exists:product_masters,id',
            'items.*.quantity' => 'required|integer|min:1',
            'items.*.unit_price' => 'required|numeric|min:0',
        ]);

        // Get customer and determine tax percent based on GST type
        $customer = Customer::findOrFail($request->customer_id);
        $taxPercent = ($customer->gst_type == 'instate') ? 18 : 18;

        // Calculate totals
        $subtotal = 0;
        foreach ($request->items as $item) {
            $subtotal += $item['quantity'] * $item['unit_price'];
        }

        $taxAmount = ($subtotal * $taxPercent) / 100;
        
        // Calculate discount from discount_percent
        $discountPercent = $request->discount_percent ?? 0;
        $discountAmount = ($subtotal * $discountPercent) / 100;
        
        $totalAmount = $subtotal + $taxAmount - $discountAmount;

        // Create quotation
        $quotation = Quotation::create([
            'quotation_number' => Quotation::generateQuotationNumber(),
            'customer_id' => $request->customer_id,
            'quotation_date' => $request->quotation_date,
            'valid_until' => $request->valid_until,
            'subtotal' => $subtotal,
            'tax_percent' => $taxPercent,
            'tax_amount' => $taxAmount,
            'discount_percent' => $discountPercent,
            'discount_amount' => $discountAmount,
            'total_amount' => $totalAmount,
            'notes' => $request->notes,
            'status' => 'draft',
        ]);

        // Create quotation items - LOOP through each product and insert one row per product
        foreach ($request->items as $item) {
            // Fetch the ProductMaster using the product_id from form
            $productMaster = \App\Models\ProductMaster::findOrFail($item['product_id']);
            
            // Create quotation item with product_master.id as foreign key
            QuotationItem::create([
                'quotation_id' => $quotation->id,
                'product_id' => $item['product_id'],  // This is product_master.id - satisfies foreign key
                'product_name' => $productMaster->product_name,
                'product_type' => $item['product_type'] ?? 'Standard',
                'quantity' => $item['quantity'],
                'unit_price' => $item['unit_price'],
                'total_price' => $item['quantity'] * $item['unit_price'],
                'description' => $item['description'] ?? null,
            ]);
        }

        // Attach terms & conditions if selected
        if ($request->has('terms_conditions') && !empty($request->terms_conditions)) {
            $termsConditionIds = array_filter($request->terms_conditions);
            if (!empty($termsConditionIds)) {
                $quotation->termsConditions()->attach($termsConditionIds);
            }
        }

        return redirect()->route('quotations.show', $quotation)
            ->with('success', 'Quotation created successfully!');
    }

    public function show(Quotation $quotation)
    {
        $quotation->load(['customer', 'items.product', 'termsConditions']);
        return view('quotations.show', compact('quotation'));
    }

    public function generatePdf(Quotation $quotation)
    {
        $quotation->load(['customer', 'items.product']);
        
        $pdf = Pdf::loadView('quotations.pdf', compact('quotation'));
        $pdf->setPaper('a4', 'portrait');
        
        return $pdf->download('quotation-' . $quotation->quotation_number . '.pdf');
    }

    public function streamPdf(Quotation $quotation)
    {
        $quotation->load(['customer', 'items.product']);
        
        $pdf = Pdf::loadView('quotations.pdf', compact('quotation'));
        $pdf->setPaper('a4', 'portrait');
        
        return $pdf->stream('quotation-' . $quotation->quotation_number . '.pdf');
    }

    public function generatePdf2(Quotation $quotation)
    {
        $quotation->load(['customer', 'items.product', 'termsConditions']);
        $termsConditions = $quotation->termsConditions;
        
        $pdf = Pdf::loadView('quotations.pdf2', compact('quotation', 'termsConditions'));
        $pdf->setPaper('a4', 'portrait');
        
        return $pdf->download('quotation-' . $quotation->quotation_number . '.pdf');
    }

    public function streamPdf2(Quotation $quotation)
    {
        $quotation->load(['customer', 'items.product', 'termsConditions']);
        $termsConditions = $quotation->termsConditions;
        
        $pdf = Pdf::loadView('quotations.pdf2', compact('quotation', 'termsConditions'));
        $pdf->setPaper('a4', 'portrait');
        
        return $pdf->stream('quotation-' . $quotation->quotation_number . '.pdf');
    }

    // API endpoints for AJAX
    public function searchCustomers(Request $request)
    {
        $search = $request->get('search', '');
        $customers = Customer::where('customer_name', 'like', "%{$search}%")
            ->limit(10)
            ->get(['id', 'customer_name', 'address', 'city', 'mobile', 'gst_no']);
        
        return response()->json($customers);
    }

    public function searchProducts(Request $request)
    {
        $search = $request->get('search', '');
        
        // Search in Product models (not ProductMaster)
        $allProducts = \App\Models\Product::with('master')
            ->whereHas('master', function($q) use ($search) {
                if ($search) {
                    $q->where('product_name', 'like', "%{$search}%");
                }
            })
            ->orWhere('product_model', 'like', "%{$search}%")
            ->limit(10)
            ->get();
        
        // Group by product model
        $grouped = $allProducts->groupBy(function ($product) {
            return $product->product_master_id . '-' . $product->product_model . '-' . $product->price;
        })->map(function ($group) {
            $first = $group->first();
            $specs = $group->map(function ($product) {
                return [
                    'spec_name' => $product->spec_name,
                    'spec_value' => $product->spec_value,
                    'spec_unit' => $product->spec_unit,
                ];
            })->toArray();
            
            return [
                'id' => $first->product_master_id,  // Return product_master_id, NOT product id
                'product_master_id' => $first->product_master_id,
                'product_model' => $first->product_model,
                'price' => $first->price,
                'master_name' => $first->master->product_name,
                'product_name' => $first->master->product_name . ' (' . $first->product_model . ')',
                'default_price' => $first->price,
                'specifications' => $specs,
            ];
        })->values();
        
        return response()->json($grouped);
    }

    public function getProductTypes()
    {
        // Get unique product master names (main products)
        $types = ProductMaster::distinct()->pluck('product_name')->filter();
        return response()->json($types);
    }
}
